<?php

namespace App\Controllers;

use App\Core\Response;
use App\Middleware\AuthMiddleware;
use App\Services\ExamEngine;
use App\Models\ExamSession;
use App\Models\UserAnswer;

class ExamController
{
    public function start(): void
    {
        $payload = AuthMiddleware::getUserFromToken();
        $userId = (int)$payload['sub'];

        $input = json_decode(file_get_contents('php://input'), true) ?? [];
        $topicId = (int)($input['topic_id'] ?? 0);
        if ($topicId <= 0) {
            Response::error('topic_id is required', 422);
        }

        $data = ExamEngine::startExamForTopic($topicId, $userId);
        Response::json($data);
    }

    public function submit($sessionId): void
    {
        $payload = AuthMiddleware::getUserFromToken();
        $userId = (int)$payload['sub'];
        $sessionId = (int)$sessionId;

        $input = json_decode(file_get_contents('php://input'), true) ?? [];
        $answers = $input['answers'] ?? [];
        if (!is_array($answers)) {
            Response::error('answers must be an array', 422);
        }

        try {
            $result = ExamEngine::submitExam($sessionId, $userId, $answers);
            Response::json(['result' => $result]);
        } catch (\Throwable $e) {
            Response::error('Failed to submit exam', 500);
        }
    }

    public function result($sessionId): void
    {
        $payload = AuthMiddleware::getUserFromToken();
        $userId = (int)$payload['sub'];
        $sessionId = (int)$sessionId;

        $session = ExamSession::findById($sessionId);
        if (!$session || (int)$session['user_id'] !== $userId) {
            Response::error('Session not found', 404);
        }

        $result = [
            'score' => (float)($session['score'] ?? 0),
            'total_correct' => (int)($session['total_correct'] ?? 0),
            'total_wrong' => (int)($session['total_wrong'] ?? 0),
            'total_not_answered' => (int)($session['total_not_answered'] ?? 0),
        ];
        $answers = UserAnswer::detailForSession($sessionId);
        Response::json(['result' => $result, 'answers' => $answers]);
    }
}

