<?php

namespace App\Models;

use App\Core\Database;
use PDO;

class ExamSession
{
    public static function create(int $examId, int $userId, \DateTimeImmutable $startTime, \DateTimeImmutable $endTime): int
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('
            INSERT INTO exam_sessions (exam_id, user_id, start_time, end_time, status)
            VALUES (:exam_id, :user_id, :start_time, :end_time, :status)
        ');
        $stmt->execute([
            'exam_id' => $examId,
            'user_id' => $userId,
            'start_time' => $startTime->format('Y-m-d H:i:s'),
            'end_time' => $endTime->format('Y-m-d H:i:s'),
            'status' => 'ongoing',
        ]);
        return (int)$pdo->lastInsertId();
    }

    public static function markCompleted(
        int $sessionId,
        float $score,
        int $totalCorrect,
        int $totalWrong,
        int $totalNotAnswered
    ): void {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('
            UPDATE exam_sessions
            SET status = "completed",
                score = :score,
                total_correct = :total_correct,
                total_wrong = :total_wrong,
                total_not_answered = :total_not_answered
            WHERE id = :id
        ');
        $stmt->execute([
            'score' => $score,
            'total_correct' => $totalCorrect,
            'total_wrong' => $totalWrong,
            'total_not_answered' => $totalNotAnswered,
            'id' => $sessionId,
        ]);
    }

    public static function findById(int $id): ?array
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('SELECT * FROM exam_sessions WHERE id = :id LIMIT 1');
        $stmt->execute(['id' => $id]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ?: null;
    }

    public static function historyByUser(int $userId): array
    {
        $pdo = Database::getConnection();
        $stmt = $pdo->prepare('
            SELECT es.id AS session_id,
                   es.score,
                   es.start_time,
                   e.name AS exam_name
            FROM exam_sessions es
            LEFT JOIN exams e ON e.id = es.exam_id
            WHERE es.user_id = :uid
              AND es.status = "completed"
            ORDER BY es.start_time DESC
        ');
        $stmt->execute(['uid' => $userId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        foreach ($rows as &$row) {
            $row['taken_at'] = $row['start_time'];
        }
        return $rows;
    }
}

