<?php

namespace App\Services;

use App\Core\Database;
use App\Models\Exam;
use App\Models\ExamSession;
use App\Models\ExamSessionQuestion;
use App\Models\Question;
use App\Models\UserAnswer;
use App\Models\UserAttemptedQuestion;
use DateTimeImmutable;
use PDO;

class ExamEngine
{
    public static function startExamForTopic(int $topicId, int $userId): array
    {
        $exam = Exam::defaultForTopic($topicId);
        $pdo = Database::getConnection();

        $primaryQuestions = Question::randomByTopicExcludingUser($topicId, $userId, $exam->total_mcq);
        $count = count($primaryQuestions);
        if ($count < $exam->total_mcq) {
            $additional = Question::randomByTopic($topicId, $exam->total_mcq - $count);
            $primaryQuestions = array_merge($primaryQuestions, $additional);
        }

        $questionIds = array_map(fn($q) => (int)$q['id'], $primaryQuestions);

        $startTime = new DateTimeImmutable('now');
        $endTime = $startTime->modify('+' . $exam->duration_minutes . ' minutes');
        $sessionId = ExamSession::create($exam->id, $userId, $startTime, $endTime);

        ExamSessionQuestion::bulkInsert($sessionId, $questionIds);

        return [
            'session' => [
                'session_id' => $sessionId,
                'exam_id' => $exam->id,
                'total_questions' => count($questionIds),
                'end_time' => $endTime->format(DATE_ATOM),
            ],
            'questions' => $primaryQuestions,
        ];
    }

    public static function submitExam(int $sessionId, int $userId, array $answers): array
    {
        $pdo = Database::getConnection();

        $session = ExamSession::findById($sessionId);
        if (!$session) {
            throw new \RuntimeException('Session not found');
        }

        $endTime = new DateTimeImmutable($session['end_time']);
        $now = new DateTimeImmutable('now');
        if ($now > $endTime) {
            // time over - still allow submission, mark as completed
        }

        UserAnswer::bulkInsert($sessionId, $answers);

        $sql = '
            SELECT q.id,
                   q.correct_option,
                   q.negative_mark,
                   ua.selected_option
            FROM exam_session_questions esq
            JOIN questions q ON q.id = esq.question_id
            LEFT JOIN user_answers ua ON ua.session_id = esq.session_id AND ua.question_id = esq.question_id
            WHERE esq.session_id = :sid
        ';
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['sid' => $sessionId]);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $totalCorrect = 0;
        $totalWrong = 0;
        $totalNotAnswered = 0;
        $score = 0.0;

        foreach ($rows as $row) {
            $correct = $row['correct_option'];
            $selected = $row['selected_option'] ?? null;
            $negative = (float)($row['negative_mark'] ?? 0.0);

            if ($selected === null) {
                $totalNotAnswered++;
            } elseif ($selected === $correct) {
                $totalCorrect++;
                $score += 1.0;
            } else {
                $totalWrong++;
                if ($negative > 0) {
                    $score -= $negative;
                }
            }
        }

        ExamSession::markCompleted($sessionId, $score, $totalCorrect, $totalWrong, $totalNotAnswered);

        $questionIds = array_map(fn($row) => (int)$row['id'], $rows);
        UserAttemptedQuestion::markAttempted($userId, $questionIds);

        return [
            'score' => $score,
            'total_correct' => $totalCorrect,
            'total_wrong' => $totalWrong,
            'total_not_answered' => $totalNotAnswered,
        ];
    }
}

